/*
*  DISCLAIMER:
*
*      THIS SOFTWARE, SOURCE CODE AND ASSOCIATED MATERIALS INCLUDING BUT NOT LIMITED TO TUTORIALS,
*      GUIDES AND COMMENTARY PROVIDED WITH THIS EXERCISE ARE ONLY DESIGNED FOR REFERENCE PURPOSES
*      TO GIVE AN EXAMPLE TO LICENSEE FOR THEIR OWN NECESSARY DEVELOPMENT OF THEIR OWN SOFTWARE AND/OR
*      APPLICATION. IT IS NOT DESIGNED FOR ANY SPECIAL PURPOSE, SERIAL PRODUCTION OR USE IN MEDICAL,
*      MILITARY, AIR CRAFT, AVIATION, SPACE OF LIFE SUPPORT EQUIPMENT.
*
*      TO THE EXTENT PERMITTED BY LAW, THE EXERCISE SOFTWARE AND/OR SOURCE CODE AND/OR AND ASSOCIATED
*      MATERIALS IS PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND AND ONLY FOR REFERENCE PURPOSES.
*
*      SYNAPTIC LABORATORIES LTD. MAKES NO WARRANTIES, EITHER EXPRESS OR IMPLIED, WITH RESPECT TO THE
*      LICENSED SOFTWARE AND/OR SOURCE CODE AND/OR ASSOCIATED MATERIALS, CONFIDENTIAL INFORMATION AND
*      DOCUMENTATION PROVIDED HEREUNDER. 
*
*      SYNAPTIC LABORATORIES LTD. SPECIFICALLY DISCLAIMS THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
*      FITNESS FOR A PARTICULAR PURPOSE AND ANY WARRANTY AGAINST INFRINGEMENT OF ANY INTELLECTUAL
*      PROPERTY RIGHT OF ANY THIRD PARTY WITH REGARD TO THE SOFTWARE, DOCUMENTATION (SCHEMATICS ETC.),
*      SOURCE CODE AND ASSOCIATED MATERIALS, CONFIDENTIAL INFORMATION AND DOCUMENTATION.
*
*      ANY USE, COMPILATION AND TESTING OF THE SOFTWARE AND/OR SOURCE CODE IS AT LICENSEE`S OWN RISK
*      AND LICENSEE IS OBLIGED TO CONDUCT EXTENSIVE TESTS TO AVOID ANY ERRORS AND FAILURE IN THE
*      COMPILED SOURCE CODE, DOCUMENTATION (SCHEMATICS ETC.) AND THE HEREFROM GENERATED SOFTWARE
*      OF LICENSEE.
*
*      EXCEPT FOR WILFULL INTENT SYNAPTIC LABORATORIES LTD. SHALL IN NO EVENT BE ENTITLED TO OR LIABLE
*      FOR ANY INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES OF ANY KIND OR NATURE, INCLUDING,
*      WITHOUT LIMITATION, BUSINESS INTERRUPTION COSTS, LOSS OF PROFIT OR REVENUE, LOSS OF DATA,
*      PROMOTIONAL OR MANUFACTURING EXPENSES, OVERHEAD, COSTS OR EXPENSES ASSOCIATED WITH WARRANTY
*      OR INTELLECTUAL PROPERTY INFRINGEMENT CLAIMS, INJURY TO REPUTATION OR LOSS OF CUSTOMERS.
*
*/



#include "sll_ca_xspi_mc_t004_reg.h"


#define FlashWriteUserCmd(b,o,d)    sll_flash_write_cmd(b, o, d)
#define FlashReadUserCmd(b,o)       sll_flash_read_cmd(b, o)


/******************************************************************************
* 
* sll_xspi_mc_reg_read  - Read 32-bit value from memory over the control port
*                         with CA format {cad_high[15:0], cad_low[31:0]}  
*
* RETURNS: 32-bit value  
*
* ERRNO  : None
*/

uint32_t sll_xspi_mc_reg_read  (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd     /* command format to issue to XSPI_MC_REG_CMD register */
)
{

  uint32_t cad_rdata_valid = 0;
  uint32_t cad_rdata;
  volatile uint32_t reg_status  = 0;
  
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO(base_csr, cad_low);
  IOWR_SLL_XSPI_MC_REG_CAD_HI(base_csr, cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, 1);
  IOWR_SLL_XSPI_MC_REG_CMD(base_csr, dev_cmd);
   
  //wait until read data is available is ready
   while (cad_rdata_valid == 0) {
    reg_status      = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    cad_rdata_valid = (reg_status & SLL_XSPI_MC_REG_STATUS_RDATA_VLD) != 0;
  }
  
  
  cad_rdata = IORD_SLL_XSPI_MC_REG_RDATA(base_csr);
  
  return (uint32_t) cad_rdata;
  
};

/******************************************************************************
* 
* sll_xspi_mc_reg_read  - Read 32-bit value from memory over the control port
*                         with CA format {cad_high[15:0], cad_low[31:0]}  
*
* RETURNS: 32-bit value  
*
* ERRNO  : None
*/

void sll_xspi_mc_reg_read2  (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd,    /* command format to issue to XSPI_MC_REG_CMD register */
  uint32_t byte_count, /* Number of bytes to read */
  uint8_t *data_buf   /* buffer containing data to program */
)
{
  int      i;
  uint32_t word_count      = 0;
  uint32_t cad_rdata_valid = 0;
  uint8_t  return_rdata_8, byte_pos;
  volatile uint32_t cad_rdata;
  volatile uint32_t reg_status  = 0;
  
   /* Set count in 32-bit words*/
   if (byte_count < 4) {
    	word_count = 1;
   } else  {
    	word_count = byte_count/4;
   }
  	
  	
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO(base_csr,  cad_low);
  IOWR_SLL_XSPI_MC_REG_CAD_HI(base_csr,  cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, word_count);
  IOWR_SLL_XSPI_MC_REG_CMD(base_csr, dev_cmd);
   
  //wait until read data is available is ready
   while (cad_rdata_valid == 0) {
    reg_status      = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    cad_rdata_valid = (reg_status & SLL_XSPI_MC_REG_STATUS_RDATA_VLD) != 0;
  }
   
  
  for (i=0; i < byte_count; i++) {
  	byte_pos = i%4;
  	//read 32-bit word every 4 bytes
  	if (byte_pos == 0) {
       cad_rdata = IORD_SLL_XSPI_MC_REG_RDATA(base_csr);
    }   

    return_rdata_8 = (cad_rdata >> (byte_pos*8)) & 0xFF;
    
    *data_buf++ = return_rdata_8;
  }

};


/******************************************************************************
* 
* sll_xspi_mc_reg_read_blk  - Read block of data from memory over the control port
*                             with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: None 
*
* ERRNO  : None
*/

void sll_xspi_mc_reg_read_blk (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd,    /* command format to issue to XSPI_MC_REG_CMD register */
  uint32_t word_count, /* number of 32-bit words to be written */
  uint32_t *data_buf   /* buffer containing data after read*/
)
{

  uint32_t cad_all_rdata_valid = 0;
  uint32_t current_offset =0;
  
  volatile uint32_t reg_status  = 0;
  
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO (base_csr, cad_low);
  IOWR_SLL_XSPI_MC_REG_CAD_HI (base_csr, cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, word_count);
  IOWR_SLL_XSPI_MC_REG_CMD    (base_csr, dev_cmd);
   
  //wait until read data is available is ready
   while (cad_all_rdata_valid == 0) {
    reg_status          = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    cad_all_rdata_valid = (reg_status & SLL_XSPI_MC_REG_STATUS_ALL_RDATA_VLD) != 0;
  }
  

  /* Set count */
  IOWR_SLL_XSPI_MC_REG_CAD_LEN (base_csr, word_count);

  /* Load Data into Buffer */
  while(current_offset <= (word_count-1))
  {

    /* Get Data into Buffer */
    *data_buf++ = IORD_SLL_XSPI_MC_REG_RDATA (base_csr);
    current_offset++;
  }

  
};



/******************************************************************************
* 
* sll_psram_reg_read  - Read 32-bit value to PSRAM(@CS1) Register over the control port
*                        with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: 32-bit value  
*
* ERRNO  : None
*/

uint32_t sll_psram_reg_read (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low     /* maps to CA[31:0] */
) 
{

  uint32_t cad_rdata_valid = 0;
  uint32_t cad_rdata;
  volatile uint32_t reg_status  = 0;
  
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO (base_csr, cad_low);
  IOWR_SLL_XSPI_MC_REG_CAD_HI (base_csr, cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, 1);
  IOWR_SLL_XSPI_MC_REG_CMD    (base_csr, SLL_XSPI_MC_RAM_CS1_RD);
  
  //wait until read data is available is ready
   while (cad_rdata_valid == 0) {
    reg_status      = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    cad_rdata_valid = (reg_status & SLL_XSPI_MC_REG_STATUS_RDATA_VLD) != 0;
  }
  
  
  cad_rdata = IORD_SLL_XSPI_MC_REG_RDATA(base_csr);
  
  return (uint32_t) cad_rdata;
  
};


/******************************************************************************
* 
* sll_xspi_mc_reg_write - Write 32-bit value to memory over the control port
*                         with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: None 
*
* ERRNO  : None
*/

void    sll_xspi_mc_reg_write (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd,     /* command format to issue to XSPI_MC_REG_CMD register */
  uint32_t wdata       /* 32-bit data to be written */
)
{

  uint32_t cad_cmd_rdy = 0;
  volatile uint32_t reg_status  = 0;
 
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO (base_csr, cad_low );
  IOWR_SLL_XSPI_MC_REG_CAD_HI (base_csr, cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, 1       );  
  IOWR_SLL_XSPI_MC_REG_WDATA32(base_csr, wdata   );
  IOWR_SLL_XSPI_MC_REG_CMD    (base_csr, dev_cmd );

  //wait until command is ready
  while (cad_cmd_rdy == 0) {
    reg_status    = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    if ((reg_status & SLL_XSPI_MC_REG_STATUS_CAD_ACTIVE) == 0) {
      cad_cmd_rdy = 1;
    }   
  }
 
}


/******************************************************************************
* 
* sll_xspi_mc_reg_write - Write 16-bit value to memory over the control port
*                         with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: None 
*
* ERRNO  : None
*/

void     sll_xspi_mc_reg_write_x16 (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd,    /* command format to issue to XSPI_MC_REG_CMD register */
  uint16_t wdata       /* 16-bit data to be written */
) 
{

  uint32_t cad_cmd_rdy = 0;
  volatile uint32_t reg_status  = 0;
  uint32_t wdata32 = ((wdata & 0xFFFF) << 16) | (wdata & 0xFFFF);
 
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO (base_csr, cad_low );
  IOWR_SLL_XSPI_MC_REG_CAD_HI (base_csr, cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, 0       );
  IOWR_SLL_XSPI_MC_REG_WDATA32(base_csr, wdata32 );
  IOWR_SLL_XSPI_MC_REG_CMD    (base_csr, dev_cmd );

  //wait until command is ready
  while (cad_cmd_rdy == 0) {
    reg_status    = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    if ((reg_status & SLL_XSPI_MC_REG_STATUS_CAD_ACTIVE) == 0) {
      cad_cmd_rdy = 1;
    }   
  }
 
}

/******************************************************************************
* 
* sll_xspi_mc_reg_write_blk - Write block fo data to memory over the control port
*                             with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: None 
*
* ERRNO  : None
*/

void    sll_xspi_mc_reg_write_blk (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd,    /* command format to issue to XSPI_MC_REG_CMD register */
  uint32_t word_count, /* number of 32-bit words to be written */
  uint32_t *data_buf   /* buffer containing data to program */
)
{
  uint32_t current_offset = 0;
  uint32_t cad_cmd_rdy    = 0;
  volatile uint32_t reg_status  = 0;

  
  /* Set count */
  IOWR_SLL_XSPI_MC_REG_CAD_LEN (base_csr, word_count);

  /* Load Data into Buffer */
  while(current_offset <= (word_count-1))
  {

    /* Write Data */
    IOWR_SLL_XSPI_MC_REG_WDATA32 (base_csr, *data_buf++);
    current_offset++;
  }

  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO (base_csr, cad_low   );
  IOWR_SLL_XSPI_MC_REG_CAD_HI (base_csr, cad_high  );
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, word_count);  
  IOWR_SLL_XSPI_MC_REG_CMD    (base_csr, dev_cmd   );

  //wait until command is ready
  while (cad_cmd_rdy == 0) {
    reg_status    = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    if ((reg_status & SLL_XSPI_MC_REG_STATUS_CAD_ACTIVE) == 0) {
      cad_cmd_rdy = 1;
    }   
  }
 
}



/******************************************************************************
* 
* sll_xspi_mc_reg_write_blk - Write block fo data to memory over the control port
*                             with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: None 
*
* ERRNO  : None
*/

void    sll_xspi_mc_reg_write_blk2 (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t dev_cmd,    /* command format to issue to XSPI_MC_REG_CMD register */
  uint32_t byte_count, /* number of 8-bit words to be written */
  uint8_t *data_buf    /* buffer containing data to program */
)
{
	int      i;
  uint32_t word_count     = 0;
  uint8_t  wdata8         = 0;
  uint32_t wdata32        = 0;
  uint32_t current_offset = 0;
  uint32_t cad_cmd_rdy    = 0;  
  volatile uint32_t reg_status  = 0;

  
  /* Set count in 32-bit words*/
   if (byte_count < 4) {
    	word_count = 1;
   } else  {
    	word_count = byte_count/4;
   }
  	
  	
  /* Set count */
  IOWR_SLL_XSPI_MC_REG_CAD_LEN (base_csr, word_count);

  /* Load Data into Buffer */
  while(current_offset < byte_count)
  {
    wdata32  =0;

    if (byte_count == 1) {
      wdata8  = *data_buf++;
      wdata32 = wdata8 | (wdata8 << 8) | (wdata8 << 16) | (wdata8 <<24);
     } else if (byte_count == 2) {
     	 //concatinate 16-bit data from 2 8-bit data
     	 for (i=0; i<2; i++) {
         wdata8   = *data_buf++;
         wdata32 |= (wdata8 << (i*8));
       }  
     	 //replicate 16-bit to create 32-bit
       wdata32 |= wdata32 << 16;
     } else {
       for (i=0; i<4; i++) {
          wdata8   = *data_buf++;
          wdata32 |= (wdata8 << (i*8));
       }    
     }
     

    /* Write Data */
    IOWR_SLL_XSPI_MC_REG_WDATA32 (base_csr, wdata32);
    current_offset += 4;
  }

  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO (base_csr, cad_low   );
  IOWR_SLL_XSPI_MC_REG_CAD_HI (base_csr, cad_high  );
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, word_count);  
  IOWR_SLL_XSPI_MC_REG_CMD    (base_csr, dev_cmd   );

  //wait until command is ready
  while (cad_cmd_rdy == 0) {
    reg_status    = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    if ((reg_status & SLL_XSPI_MC_REG_STATUS_CAD_ACTIVE) == 0) {
      cad_cmd_rdy = 1;
    }   
  }
 
}

/******************************************************************************
* 
* sll_psram_reg_write  - Write 32-bit value to PSRAM(@CS1) Register over the control port
*                        with CA format {cad_high[15:0], cad_low[31:0]} 
*
* RETURNS: None 
*
* ERRNO  : None
*/
void    sll_psram_reg_write (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t cad_high,   /* maps to CA[47:32] */
  uint32_t cad_low,    /* maps to CA[31:0] */
  uint32_t wdata       /* 32-bit data to be written */
) 
{

  uint32_t cad_cmd_rdy = 0;
  volatile uint32_t reg_status  = 0;
 
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_LO  (base_csr, cad_low);
  IOWR_SLL_XSPI_MC_REG_CAD_HI  (base_csr, cad_high);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN (base_csr, 0);
  IOWR_SLL_XSPI_MC_REG_WDATA32 (base_csr, wdata);
  IOWR_SLL_XSPI_MC_REG_CMD     (base_csr, SLL_XSPI_MC_RAM_CS1_WR);

  //wait until command is ready
  while (cad_cmd_rdy == 0) {
    reg_status    = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
    if ((reg_status & SLL_XSPI_MC_REG_STATUS_CAD_ACTIVE) == 0) {
      cad_cmd_rdy = 1;
    }   
  }
 
}


/******************************************************************************
* 
* sll_flash_write_cmd  - Write 16-bit value to Flash(@CS0) over the control port
*                        with address offset 
*
* RETURNS: None 
*
* ERRNO  : None
*/

void    sll_flash_write_cmd (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t offset,     /* address offset from base address (byte aligned) */
  uint16_t wdata       /* 16-bit data to be written */
) 
{

  uint32_t cad_cmd_rdy = 0;
  volatile uint32_t reg_status  = 0;
  uint32_t wdata32 = ((wdata & 0xFFFF) << 16) | (wdata & 0xFFFF);

  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_HI  (base_csr, 0x0000);
  IOWR_SLL_XSPI_MC_REG_CAD_LO  (base_csr, offset);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN (base_csr, 0);
  IOWR_SLL_XSPI_MC_REG_WDATA32 (base_csr, wdata32);
  IOWR_SLL_XSPI_MC_REG_CMD     (base_csr, SLL_XSPI_MC_FLASH_CS0_WR);
  
  //wait until command is ready
  while (cad_cmd_rdy == 0) {
     reg_status   = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
     if ((reg_status & SLL_XSPI_MC_REG_STATUS_CAD_ACTIVE) == 0) {
      cad_cmd_rdy = 1;
    }   
  }
}

/******************************************************************************
* 
* sll_flash_read_cmd  - Read 16-bit value from Flash(@CS0) over the control port
*                        with address offset 
*
* RETURNS: 16-bit value  
*
* ERRNO  : None
*/

uint16_t sll_flash_read_cmd (
  uint32_t base_csr,   /* control port base address in system */
  uint32_t offset      /* address offset from base address (byte aligned) */
) 
{

  uint32_t cad_rdata_valid = 0;
  uint16_t cad_rdata;
  volatile uint32_t reg_status  = 0;
  
  //send command/address/data field
  IOWR_SLL_XSPI_MC_REG_CAD_HI(base_csr, 0x0000);
  IOWR_SLL_XSPI_MC_REG_CAD_LO(base_csr, offset);
  IOWR_SLL_XSPI_MC_REG_CAD_LEN(base_csr, 1);
  IOWR_SLL_XSPI_MC_REG_CMD(base_csr, SLL_XSPI_MC_FLASH_CS0_RD);
   
  //wait until read data is available is ready
  while (cad_rdata_valid == 0) {
     reg_status      = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
     cad_rdata_valid = (reg_status & SLL_XSPI_MC_REG_STATUS_RDATA_VLD) != 0;
  }
  
  cad_rdata = IORD_SLL_XSPI_MC_REG_RDATA(base_csr);
  
  return (uint16_t) cad_rdata;
  
};  

/******************************************************************************
* 
* sll_xspi_mc_wait_pwrup_rdy  Check xSPI-MC PowerUP ready routine
*
*
* RETURNS: OK                (0) or 
*          error on timoeout (1)
*
* ERRNO  : 1
*/
uint32_t sll_xspi_mc_wait_pwrup_rdy (
uint32_t base_csr  /* control port base address in system */
) 
{

  volatile uint32_t reg_status  = 0;
  uint32_t pwrup_done  = 0;
  uint32_t timout_cntr = 0x0FFFFF;
  
   
  //wait until read data is available is ready
  while (pwrup_done == 0) {
     reg_status      = IORD_SLL_XSPI_MC_REG_STATUS(base_csr);
     pwrup_done      = (reg_status & SLL_XSPI_MC_REG_STATUS_POWER_UP) == 0;
     
     /* sleep 10 us and check again */
     SLL_DelayMicroseconds (10);
          
     timout_cntr--;
     /* timout  error */
     if (timout_cntr == 0) {
       return 1;
     }
  }
    
  return 0 ;
  
}; 


/******************************************************************************
* 
* DelayMicroseconds - Performs a delay.  If you have a better way,
*                     
* RETURNS: void
*
*/

void SLL_DelayMicroseconds(int microseconds)
{
	 SLEEP_US(microseconds);
}



